use Test::More;
use strict;
use warnings;
our ( $sub, $globref );

BEGIN {
	use_ok('Hades::Realm::Exporter');
	$sub     = sub { };
	$globref = \*globref;
}
subtest 'new' => sub {
	plan tests => 9;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	ok( $obj = Hades::Realm::Exporter->new(),
		q{$obj = Hades::Realm::Exporter->new()}
	);
	isa_ok( $obj, 'Hades::Realm::Exporter' );
	ok( $obj = Hades::Realm::Exporter->new( {} ),
		q{$obj = Hades::Realm::Exporter->new({})}
	);
	ok( $obj = Hades::Realm::Exporter->new(),
		q{$obj = Hades::Realm::Exporter->new()}
	);
	is_deeply( $obj->export, {}, q{$obj->export} );
	ok( $obj
		    = Hades::Realm::Exporter->new(
			{ export => { 'test' => 'test' } } ),
		q{$obj = Hades::Realm::Exporter->new({ export => { 'test' => 'test' } })}
	);
	eval { $obj = Hades::Realm::Exporter->new( { export => [] } ) };
	like(
		$@,
		qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::Exporter->new({ export => [] })}
	);
	eval { $obj = Hades::Realm::Exporter->new( { export => 'algea' } ) };
	like(
		$@,
		qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::Exporter->new({ export => 'algea' })}
	);
};
subtest 'export' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'export' );
	is_deeply(
		$obj->export( { 'test' => 'test' } ),
		{ 'test' => 'test' },
		q{$obj->export({ 'test' => 'test' })}
	);
	eval { $obj->export( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->export([])} );
	eval { $obj->export('algea') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->export('algea')} );
	is_deeply( $obj->export, { 'test' => 'test' }, q{$obj->export} );
};
subtest 'default_export_hash' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'default_export_hash' );
	eval {
		$obj->default_export_hash(
			[],
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash([], { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash(
			'geras',
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash('geras', { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			[], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			'thanatos', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), 'thanatos', { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			{ 'test' => 'test' }, [] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), { 'test' => 'test' }, [])}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			{ 'test' => 'test' }, 'gaudia' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), { 'test' => 'test' }, 'gaudia')}
	);
};
subtest 'build_new' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_new' );
	eval { $obj->build_new( [], { 'test' => 'test' }, 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new([], { 'test' => 'test' }, 'aporia')}
	);
	eval { $obj->build_new( 'aporia', { 'test' => 'test' }, 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new('aporia', { 'test' => 'test' }, 'aporia')}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), [], 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), [], 'aporia')}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), 'phobos', 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), 'phobos', 'aporia')}
	);
};
subtest 'build_exporter' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_exporter' );
	eval {
		$obj->build_exporter(
			[],
			bless( {}, 'Test' ),
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter([], bless({}, 'Test'), { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			\1,
			bless( {}, 'Test' ),
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter(\1, bless({}, 'Test'), { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			'nosoi', [],
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('nosoi', [], { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			'nosoi', 'penthos',
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('nosoi', 'penthos', { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter( 'nosoi', bless( {}, 'Test' ),
			[], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('nosoi', bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter( 'nosoi', bless( {}, 'Test' ),
			'nosoi', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('nosoi', bless({}, 'Test'), 'nosoi', { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			'nosoi',
			bless( {}, 'Test' ),
			{ 'test' => 'test' }, []
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('nosoi', bless({}, 'Test'), { 'test' => 'test' }, [])}
	);
	eval {
		$obj->build_exporter(
			'nosoi',
			bless( {}, 'Test' ),
			{ 'test' => 'test' }, 'thanatos'
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('nosoi', bless({}, 'Test'), { 'test' => 'test' }, 'thanatos')}
	);
};
subtest 'build_export_tags' => sub {
	plan tests => 12;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_export_tags' );
	eval {
		$obj->build_export_tags( [], 'penthos', { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags([], 'penthos', { 'test' => 'test' }, undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( \1, 'penthos', { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags(\1, 'penthos', { 'test' => 'test' }, undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'algea', [], { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', [], { 'test' => 'test' }, undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'algea', \1, { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', \1, { 'test' => 'test' }, undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'algea', 'penthos', [], undef, ['test'] );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', 'penthos', [], undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'algea', 'penthos', 'thanatos', undef,
			['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', 'penthos', 'thanatos', undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'algea', 'penthos', { 'test' => 'test' },
			[], ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', 'penthos', { 'test' => 'test' }, [], ['test'])}
	);
	eval {
		$obj->build_export_tags( 'algea', 'penthos', { 'test' => 'test' },
			'nosoi', ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', 'penthos', { 'test' => 'test' }, 'nosoi', ['test'])}
	);
	eval {
		$obj->build_export_tags( 'algea', 'penthos', { 'test' => 'test' },
			undef, {} );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', 'penthos', { 'test' => 'test' }, undef, {})}
	);
	eval {
		$obj->build_export_tags( 'algea', 'penthos', { 'test' => 'test' },
			undef, 'limos' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('algea', 'penthos', { 'test' => 'test' }, undef, 'limos')}
	);
};
subtest 'after_class' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'after_class' );
	eval { $obj->after_class( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class([])} );
	eval { $obj->after_class('thanatos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class('thanatos')}
	);
};
subtest 'build_sub_or_accessor_attributes' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub_or_accessor_attributes' );
};
subtest 'build_accessor_no_arguments' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_accessor_no_arguments' );
};
subtest 'build_accessor_code' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_accessor_code' );
	eval { $obj->build_accessor_code( [], 'limos', 'geras', 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code([], 'limos', 'geras', 'hypnos')}
	);
	eval { $obj->build_accessor_code( \1, 'limos', 'geras', 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code(\1, 'limos', 'geras', 'hypnos')}
	);
	eval { $obj->build_accessor_code( 'aporia', [], 'geras', 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('aporia', [], 'geras', 'hypnos')}
	);
	eval { $obj->build_accessor_code( 'aporia', \1, 'geras', 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('aporia', \1, 'geras', 'hypnos')}
	);
	eval { $obj->build_accessor_code( 'aporia', 'limos', [], 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('aporia', 'limos', [], 'hypnos')}
	);
	eval { $obj->build_accessor_code( 'aporia', 'limos', \1, 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('aporia', 'limos', \1, 'hypnos')}
	);
	eval { $obj->build_accessor_code( 'aporia', 'limos', 'geras', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('aporia', 'limos', 'geras', [])}
	);
	eval { $obj->build_accessor_code( 'aporia', 'limos', 'geras', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('aporia', 'limos', 'geras', \1)}
	);
};
subtest 'build_accessor' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_accessor' );
};
subtest 'build_modify' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_modify' );
};
subtest 'build_sub_no_arguments' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub_no_arguments' );
};
subtest 'build_sub_code' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub_code' );
	eval { $obj->build_sub_code( [], 'geras', 'thanatos', 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code([], 'geras', 'thanatos', 'nosoi')}
	);
	eval { $obj->build_sub_code( \1, 'geras', 'thanatos', 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code(\1, 'geras', 'thanatos', 'nosoi')}
	);
	eval { $obj->build_sub_code( 'thanatos', [], 'thanatos', 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('thanatos', [], 'thanatos', 'nosoi')}
	);
	eval { $obj->build_sub_code( 'thanatos', \1, 'thanatos', 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('thanatos', \1, 'thanatos', 'nosoi')}
	);
	eval { $obj->build_sub_code( 'thanatos', 'geras', [], 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('thanatos', 'geras', [], 'nosoi')}
	);
	eval { $obj->build_sub_code( 'thanatos', 'geras', \1, 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('thanatos', 'geras', \1, 'nosoi')}
	);
	eval { $obj->build_sub_code( 'thanatos', 'geras', 'thanatos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('thanatos', 'geras', 'thanatos', [])}
	);
	eval { $obj->build_sub_code( 'thanatos', 'geras', 'thanatos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('thanatos', 'geras', 'thanatos', \1)}
	);
};
subtest 'build_sub' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub' );
};
subtest 'build_clearer' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_clearer' );
	eval { $obj->build_clearer( [], 'aporia', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer([], 'aporia', { 'test' => 'test' })}
	);
	eval { $obj->build_clearer( 'gaudia', 'aporia', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer('gaudia', 'aporia', { 'test' => 'test' })}
	);
	eval {
		$obj->build_clearer( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_clearer( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_clearer( bless( {}, 'Test' ), 'aporia', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), 'aporia', [])}
	);
	eval { $obj->build_clearer( bless( {}, 'Test' ), 'aporia', 'geras' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), 'aporia', 'geras')}
	);
};
subtest 'build_predicate' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_predicate' );
	eval { $obj->build_predicate( [], 'aporia', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate([], 'aporia', { 'test' => 'test' })}
	);
	eval {
		$obj->build_predicate( 'penthos', 'aporia', { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate('penthos', 'aporia', { 'test' => 'test' })}
	);
	eval {
		$obj->build_predicate( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_predicate( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_predicate( bless( {}, 'Test' ), 'aporia', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), 'aporia', [])}
	);
	eval { $obj->build_predicate( bless( {}, 'Test' ), 'aporia', 'curae' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), 'aporia', 'curae')}
	);
};
subtest 'build_coerce' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_coerce' );
	eval { $obj->build_coerce( [], 'aporia', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce([], 'aporia', undef)}
	);
	eval { $obj->build_coerce( \1, 'aporia', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce(\1, 'aporia', undef)}
	);
	eval { $obj->build_coerce( 'algea', [], undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('algea', [], undef)}
	);
	eval { $obj->build_coerce( 'algea', \1, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('algea', \1, undef)}
	);
	eval { $obj->build_coerce( 'algea', 'aporia', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('algea', 'aporia', [])}
	);
	eval { $obj->build_coerce( 'algea', 'aporia', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('algea', 'aporia', \1)}
	);
};
subtest 'build_trigger' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_trigger' );
	eval { $obj->build_trigger( [], 'hypnos', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger([], 'hypnos', undef)}
	);
	eval { $obj->build_trigger( \1, 'hypnos', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger(\1, 'hypnos', undef)}
	);
	eval { $obj->build_trigger( 'algea', [], undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('algea', [], undef)}
	);
	eval { $obj->build_trigger( 'algea', \1, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('algea', \1, undef)}
	);
	eval { $obj->build_trigger( 'algea', 'hypnos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('algea', 'hypnos', [])}
	);
	eval { $obj->build_trigger( 'algea', 'hypnos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('algea', 'hypnos', \1)}
	);
};
subtest 'build_tests' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_tests' );
	eval { $obj->build_tests( [], { 'test' => 'test' }, undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests([], { 'test' => 'test' }, undef, undef)}
	);
	eval { $obj->build_tests( \1, { 'test' => 'test' }, undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests(\1, { 'test' => 'test' }, undef, undef)}
	);
	eval { $obj->build_tests( 'aporia', [], undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('aporia', [], undef, undef)}
	);
	eval { $obj->build_tests( 'aporia', 'geras', undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('aporia', 'geras', undef, undef)}
	);
	eval { $obj->build_tests( 'aporia', { 'test' => 'test' }, [], undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('aporia', { 'test' => 'test' }, [], undef)}
	);
	eval { $obj->build_tests( 'aporia', { 'test' => 'test' }, \1, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('aporia', { 'test' => 'test' }, \1, undef)}
	);
	eval { $obj->build_tests( 'aporia', { 'test' => 'test' }, undef, [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('aporia', { 'test' => 'test' }, undef, [])}
	);
	eval {
		$obj->build_tests( 'aporia', { 'test' => 'test' }, undef, 'gaudia' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('aporia', { 'test' => 'test' }, undef, 'gaudia')}
	);
};
done_testing();
